
//Adds code editing support to textarea, tab indent, auto indent, auto brackets and syntax highlighting.
/**@constructor*/
function CodeEditor(id, indentStr,autoIndent,autoBrackets, hlId,highlighting,hlDelay) {
	//textarea element
	this.text = document.getElementById(id);
	this.text.codeEditorAttachment = this;
	
	//syntaxhighlighting and options
	this.highlight = highlighting?document.getElementById(hlId):null;
	this.highlighting = highlighting;
	this.highlightDelay = hlDelay?hlDelay:0;
	this.highlightTimeoutRunning = false;
	this.highlightLastChange = 0;
	
	//code input options
	this.indentString = indentStr;
	this.autoIndent = autoIndent;
	this.autoBrackets = autoBrackets;
	this.codeLastChanged = 0;
	
	//setup event listeners
	this.text.addEventListener("keydown", OnCodeEditorKeyDown);
	this.text.addEventListener("input", OnCodeEditorInput);
	if (highlighting) this.text.addEventListener("scroll", OnCodeEditorScroll);
}

//manually update syntax highlighting
CodeEditor.prototype.UpdateSyntaxHighlighting = function() {
	if (this.highlighting) this.highlight.innerHTML = this.highlighting.Highlight(this.text.value,true)+"\n";
}

//set text
CodeEditor.prototype.SetText = function(txt,dontUpdateHighlight) {
	this.text.value = txt;
	if (this.highlighting && !dontUpdateHighlight) this.highlight.innerHTML = this.highlighting.Highlight(txt,true)+"\n";
}


//update highlighting on a delay
function CodeEditorUpdateHighlight(cedit) {
	if (!cedit.highlighting || cedit.highlightDelay <= 0) {
		cedit.highlightTimeoutRunning = false;
		return;
	}
	
	if (cedit.codeLastChanged > cedit.highlightLastChange) {
		var tarea = cedit.text, hl = cedit.highlight;
		hl.innerHTML = cedit.highlighting.Highlight(tarea.value,true)+"\n";
		hl.scrollLeft = tarea.scrollLeft;
		hl.scrollTop = tarea.scrollTop;
		
		cedit.highlightLastChange = cedit.codeLastChanged;
	}
	
	setTimeout(CodeEditorUpdateHighlight, cedit.highlightDelay, cedit);
}


//on code changed
function OnCodeEditorInput(e) {
	var tarea = e.target, cedit = tarea.codeEditorAttachment;
	cedit.codeLastChanged = e.timeStamp;
	
	//update highlighting
	if (cedit.highlighting) {	
		if (cedit.highlightDelay === 0) {
			var hl = cedit.highlight;
			hl.innerHTML = cedit.highlighting.Highlight(tarea.value,true)+"\n";
			hl.scrollLeft = tarea.scrollLeft;
			hl.scrollTop = tarea.scrollTop;
		
		} else {
			if (!cedit.highlightTimeoutRunning) {
				cedit.highlightTimeoutRunning = true;
				setTimeout(CodeEditorUpdateHighlight,0,cedit);
			}
		}
	}
}


//custom textarea input keys
function OnCodeEditorKeyDown(e) {
	var tarea = e.target, cedit = tarea.codeEditorAttachment;
	
	if (cedit.autoBrackets) {
		if (e.key === "[") {
			//insert brackets
			var start = tarea.selectionStart;
			tarea.value = tarea.value.substring(0,start)+"[]"+tarea.value.substring(tarea.selectionEnd,tarea.value.length);
			tarea.selectionStart = tarea.selectionEnd = start+1;
			e.preventDefault();
			OnCodeEditorInput(e);
			return;
		
		} else if (e.key === "{") {
			//insert curly braces
			var start = tarea.selectionStart;
			tarea.value = tarea.value.substring(0,start)+"{}"+tarea.value.substring(tarea.selectionEnd,tarea.value.length);
			tarea.selectionStart = tarea.selectionEnd = start+1;
			e.preventDefault();
			OnCodeEditorInput(e);
			return;
			
		} else if (e.key === "(") {
			//insert parantheses
			var start = tarea.selectionStart;
			tarea.value = tarea.value.substring(0,start)+"()"+tarea.value.substring(tarea.selectionEnd,tarea.value.length);
			tarea.selectionStart = tarea.selectionEnd = start+1;
			e.preventDefault();
			OnCodeEditorInput(e);
			return;
		}
	}
	
	if (cedit.autoIndent) {
		if (e.code === "Enter" || e.code === "NumpadEnter") {//enter newline with auto indent
			var start = tarea.selectionStart,
				scan = start, indent = "", newScope = 0;
		
			//scan backwards to last new line to detect number of tab indents
			while (--scan >= 0) {
				var ch = tarea.value.charCodeAt(scan);
				if (ch === 10) break;//new line, were done, 10 = \n
			
				if (ch === 20 || ch === 9) {//20 = space, 9 = tab
					indent += String.fromCharCode(ch);//measure whitespace before newline
				} else {
					indent = "";
					if (newScope === 0) {
						if (ch === 125) newScope = -1;//125 = }
						if (ch === 123) newScope = 1;//123 = {
					}
				}
			}
		
			if (newScope === 1 && start !== tarea.value.length && tarea.value.charCodeAt(start) === 125) {
				//if inside {}, add 2 new lines and position cursor between them
				tarea.value = tarea.value.substring(0,start)+"\n"+indent+cedit.indentString+"\n"+indent+tarea.value.substring(tarea.selectionEnd,tarea.value.length);
				tarea.selectionStart = tarea.selectionEnd = start+2+indent.length;
			
			} else {
				if (newScope === 1) indent += cedit.indentString;
		
				//insert new line and indents
				tarea.value = tarea.value.substring(0,start)+"\n"+indent+tarea.value.substring(tarea.selectionEnd,tarea.value.length);
				tarea.selectionStart = tarea.selectionEnd = start+1+indent.length;
			}
		
			e.preventDefault();
			OnCodeEditorInput(e);
			return;
		}
	}

	if (e.code === "Tab") {//tab to indent
		var start = tarea.selectionStart,
			end = tarea.selectionEnd;
		if (start === end) {//insert tab
			tarea.value = tarea.value.substring(0,start)+cedit.indentString+tarea.value.substring(end,tarea.value.length);
			tarea.selectionStart = tarea.selectionEnd = start+cedit.indentString.length;
			
		} else {//tab selected text
			var ins;
			if (e.shiftKey) {
				//remove tabs
				var szero = (start===0);
				ins = tarea.value.substring(szero ? 0 : start-1, end).replaceAll("\n"+cedit.indentString,"\n");
				if (szero) ins = ins.substring(1);

			} else {
				ins = cedit.indentString;
				for (var i = start; i < end; i++) {
					var ch = tarea.value[i];
					ins += ch;
					if (ch === "\n" && i < end-1) ins += cedit.indentString;
				}
			}
			tarea.value = tarea.value.substring(0,start)+ins+tarea.value.substring(end,tarea.value.length);
			tarea.selectionStart = start;
			tarea.selectionEnd = end+ins.length-(end-start);
		}
		e.preventDefault();
		OnCodeEditorInput(e);
	}
}


//match highlight scroll to textarea
function OnCodeEditorScroll(e) {
	var tarea = e.target, hl = tarea.codeEditorAttachment.highlight;
	if (hl) {
		hl.scrollLeft = tarea.scrollLeft;
		hl.scrollTop = tarea.scrollTop;
	}
}
