//Ethan Alexander Shulman 2022
/*File: IndexedStorage.js
IndexedDB and localStorage wrapper, allowing easy loading and saving of indexed text data.*/


if (!window.indexedDB) window.indexedDB = window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;


/*Property: indexedStorageDBName
*String* Currently open database name, if null no database is open.*/
var indexedStorageDBName = null,
/*Property: indexedStorageDB
*IDBDatabase* Currently open database if IndexedDB supported.*/
indexedStorageDB = null;



/*Function: IndexedStorage_Open
Open database.

Parameters:
*String* database - Database name.
*function* onReady - On ready callback, this function is called when the database is opened and ready.*/ 
function IndexedStorage_Open(database, onReady) {
	indexedStorageDBName = database;
	if (window.indexedDB) {
		var req = window.indexedDB.open(database,1);
		req.idbcallback = onReady;
		req.onsuccess = onIndexedDBOpen;
		req.onerror = onIndexedDBOpenError;
		req.onupgradeneeded = onIndexedDBUpgrade;
	} else {
		onReady();
	}
}
//internal callbacks
function onIndexedDBOpenError(e) {
	indexedStorageDB = null;
	e.target.idbcallback();
}
function onIndexedDBOpen(e) {
	indexedStorageDB = e.target.result;
	e.target.idbcallback();
}
function onIndexedDBUpgrade(e) {
	e.target.result.createObjectStore("o");
}

/*Function: IndexedStorage_Close
Close currently open database.*/
function IndexedStorage_Close() {
	if (indexedStorageDB) {
		indexedStorageDB.close();
		indexedStorageDB = null;
	}
	indexedStorageDBName = null;
}


/*Function: IndexedStorage_Load
Load text from index 'id'.

Parameters:
*String* id - Database index name.
*function* onReady - On ready callback called once operation complete, arg 0 is id, arg 1 is the loaded text data or null if failed.
On ready is called even on failure, check if arg 1 is null to check if failure.*/
function IndexedStorage_Load(id, onReady) {
	if (indexedStorageDB) {
		var t = indexedStorageDB.transaction(["o"]),
			r = t.objectStore("o").get(id);
		r.idbcallback = onReady;
		r.idbdata = id;
		r.onsuccess = onIndexedDBLoadComplete;
		r.onerror = onIndexedDBLoadError;
	} else {
		var d = null;
		try {
			d = window.localStorage[indexedStorageDBName+id];
		} catch (e) {}
		onReady(id,d);
	}
}
function onIndexedDBLoadError(e) {
	e.target.idbcallback(e.target.idbdata);
}
function onIndexedDBLoadComplete(e) {
	e.target.idbcallback(e.target.idbdata, e.target.result);
}


/*Function: IndexedStorage_Store
Store text at index 'id'.

Parameters:
*String* id - Database index name.
*String* txt - Text string to store.
*function* onReady - On ready callback called once operation complete, arg 0 is id, arg 1 is true if success or false if failed.*/
function IndexedStorage_Store(id, txt, onReady) {
	if (indexedStorageDB) {
		var t = indexedStorageDB.transaction(["o"],"readwrite");
		t.idbcallback = onReady;
		t.idbdata = id;
		t.oncomplete = onIndexedDBStoreComplete;
		t.onerror = onIndexedDBStoreError;
		t.objectStore("o").put(txt, id);
	} else {
		try {
			window.localStorage[indexedStorageDBName+id] = txt;
			if (onReady) onReady(id,true);
		} catch (e) {
			if (onReady) onReady(id,false);
		}
	}
}
function onIndexedDBStoreComplete(e) {
	if (e.target.idbcallback) e.target.idbcallback(e.target.idbdata, true);
}
function onIndexedDBStoreError(e) {
	if (e.target.idbcallback) e.target.idbcallback(e.target.idbdata, false);
}


/*Function: IndexedStorage_Delete
Delete indexed data at 'id'.

Parameters:
*String* id - Database index name.
*function* onReady - On ready callback called once operation complete, arg 0 is id, arg 1 is true if success or false if failed.*/
function IndexedStorage_Delete(id, onReady) {
	if (indexedStorageDB) {
		var t = indexedStorageDB.transaction(["o"],"readwrite");
		t.idbcallback = onReady;
		t.idbdata = id;
		t.oncomplete = onIndexedDBStoreComplete;
		t.onerror = onIndexedDBStoreError;
		t.objectStore("o").delete(id);
	} else {
		try {
			delete window.localStorage[indexedStorageDBName+id];
			onReady(id,true);
		} catch (e) {
			onReady(id,false);
		}
	}
}
