
//Quick WebGL provides simple and easy functions for doing shader processing behind the scenes(rendering to fullscreen quad)
var gl,quickGLCanvas;



//Helper function for setting texture filtering options.
function SetTextureOptions(linear,mipmap,repeatx,repeaty) {
	var mgf, mif;
	if (linear) {
		mgf = gl.LINEAR;
		mif = mipmap?gl.LINEAR_MIPMAP_LINEAR:gl.LINEAR;
	} else {
		mgf = gl.NEAREST;
		mif = mipmap?gl.NEAREST_MIPMAP_LINEAR:gl.NEAREST;
	}
	gl.texParameteri(gl.TEXTURE_2D,gl.TEXTURE_MAG_FILTER,mgf);
	gl.texParameteri(gl.TEXTURE_2D,gl.TEXTURE_MIN_FILTER,mif);
	gl.texParameteri(gl.TEXTURE_2D,gl.TEXTURE_WRAP_S,repeatx?gl.REPEAT:gl.CLAMP_TO_EDGE);
	gl.texParameteri(gl.TEXTURE_2D,gl.TEXTURE_WRAP_T,repeaty?gl.REPEAT:gl.CLAMP_TO_EDGE);
}


var boundTextures = [];

//Bind texture tex to uniform location uni.
function BindTexture(uni,tex) {
	var ni = boundTextures.indexOf(gl.INVALID_VALUE);
	if (ni === -1) ni = boundTextures.push(uni)-1;
	else boundTextures[ni] = uni;
	gl.activeTexture(gl.TEXTURE0+ni);
	gl.bindTexture(gl.TEXTURE_2D,tex);
	gl.uniform1i(uni,ni);
}

//Unbind texture at uniform location uni.
function UnbindTexture(uni) {
	boundTextures[boundTextures.indexOf(uni)] = gl.INVALID_VALUE;
}

//Unbind all textures.
function UnbindAllTextures() {
	for (var i = 0; i < boundTextures.length; i++) boundTextures[i] = gl.INVALID_VALUE;
}



var shaders = [], shaderGlobals = {}, currentShader = null;

//Helper function for compiling vertex/fragment webgl shader.
function CompileShader(src,type) {
	var s = gl.createShader(type);
	gl.shaderSource(s,src);
	gl.compileShader(s);
	if (!gl.getShaderParameter(s,gl.COMPILE_STATUS)) {
		console.error("Error compiling GLSL shader: "+gl.getShaderInfoLog(s)+"\nFrom shader:");
		console.log(src);
		gl.deleteShader(s);
		return null;
	}
	return s;
}



//Rendertexture class, easy to use framebuffer object interface
var display, currentRenderTexture = null, currentRenderTextureX, currentRenderTextureY, currentRenderTextureW, currentRenderTextureH;

/**Create new webgl rendertexture(framebuffer) w width and h height in pixels
@constructor*/
function RenderTexture(w,h,internalFormat,sourceFormat,type,depth,isDisplay) {
	this.framebuffer = isDisplay ? null : gl.createFramebuffer();
	this.texture = isDisplay ? null : gl.createTexture();
	this.depthBuffer = null;
	this.width = w;
	this.height = h;
	this.internalFormat = internalFormat;
	this.sourceFormat = sourceFormat;
	this.type = type;
	
	if (isDisplay) return
	
	gl.bindTexture(gl.TEXTURE_2D,this.texture);
	gl.texImage2D(gl.TEXTURE_2D,0,internalFormat,w,h,0,sourceFormat,type,null);
	SetTextureOptions(false,false,false,false);
	gl.bindFramebuffer(gl.FRAMEBUFFER,this.framebuffer);
	gl.framebufferTexture2D(gl.FRAMEBUFFER,gl.COLOR_ATTACHMENT0,gl.TEXTURE_2D,this.texture,0);
	currentRenderTexture = this;
	currentRenderTextureX = currentRenderTextureY = 0;
	currentRenderTextureW = w; currentRenderTextureH = h;
	
	if (depth) {
		this.depthBuffer = gl.createRenderbuffer();
		gl.bindRenderbuffer(gl.RENDERBUFFER, this.depthBuffer);
		gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_COMPONENT, w, h);
		gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.RENDERBUFFER, this.depthBuffer);
	}
}

//Resize render texture.
RenderTexture.prototype.SetSize = function(w,h,dat) {
	this.width = w;
	this.height = h;
	if (!this.framebuffer) return;
	
	gl.bindTexture(gl.TEXTURE_2D,this.texture);
	gl.texImage2D(gl.TEXTURE_2D,0,this.internalFormat,w,h,0,this.sourceFormat,this.type,dat);
	if (this.depthBuffer) {
		gl.bindRenderbuffer(gl.RENDERBUFFER, this.depthBuffer);
		gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_COMPONENT, w, h);
	}
}

//Bind sub window(x,y,w,h) of rendertexture as the current rendertarget.
RenderTexture.prototype.Bind = function(x,y,w,h) {
	if (arguments.length < 4) {
		x = y = 0;
		w = this.width; h = this.height;
	}
	if (currentRenderTexture === this && currentRenderTextureX === x && currentRenderTextureY === y && currentRenderTextureW === w && currentRenderTextureH === h) return;
	currentRenderTexture = this;
	currentRenderTextureX = x; currentRenderTextureY = y;
	currentRenderTextureW = w; currentRenderTextureH = h;
	
	gl.bindFramebuffer(gl.FRAMEBUFFER,this.framebuffer);
	gl.viewport(x,y,w,h);
	gl.scissor(x,y,w,h);
}

//Delete rendertexture freeing data.
RenderTexture.prototype.Delete = function() {
	if (!this.framebuffer) return;
	gl.deleteFramebuffer(this.framebuffer);
	gl.deleteTexture(this.texture);
	if (this.depthBuffer) gl.deleteRenderbuffer(this.depthBuffer);
}



//create mesh from array of vertex positions of size dimensions(max 4), returns array buffer
function CreateMesh(dat,size) {
	var buf = gl.createBuffer();
	gl.bindBuffer(gl.ARRAY_BUFFER,buf);
	gl.bufferData(gl.ARRAY_BUFFER,buf.constructor.name==="Float32Array"?dat:new Float32Array(dat),gl.STATIC_DRAW);
	buf.meshVertexSize = size;
	buf.meshVertexCount = dat.length/size;
	return buf;
}
function UpdateMesh(buf,dat,size,dynamic) {
	gl.bindBuffer(gl.ARRAY_BUFFER,buf);
	gl.bufferData(gl.ARRAY_BUFFER,dat,dynamic?gl.DYNAMIC_DRAW:gl.STATIC_DRAW);
	buf.meshVertexSize = size;
	buf.meshVertexCount = dat.length/size;
}
function BindMesh(buf) {
	gl.bindBuffer(gl.ARRAY_BUFFER,buf);
	gl.vertexAttribPointer(0,buf.meshVertexSize,gl.FLOAT,false,buf.meshVertexSize*4,0);
}
function DeleteMesh(buf) {
	gl.deleteBuffer(buf);
}


//Initialize quick WebGL canvas and gl context, returns false if failed
var fullscreenTriangle;
function InitializeQuickWebGL(canvas,opt) {
	quickGLCanvas = canvas;
	quickGLCanvas.width = quickGLCanvas.height = 1;
	
	//create webgl context
	var gopt = {"alpha":false, "antialias":false, "depth":false, "failIfMajorPerformanceCaveat":false, "preserveDrawingBuffer":false};
	if (opt) Object.assign(gopt,opt);
	gl = quickGLCanvas.getContext("webgl2",gopt);
	if (!gl) return false;
	
	display = new RenderTexture(4,4,gl.RGBA,gl.RGBA,gl.UNSIGNED_BYTE,false,true);
	gl.enable(gl.SCISSOR_TEST);
	gl.enableVertexAttribArray(0);
	
	//load fullscreen triangle mesh for rendering
	fullscreenTriangle = CreateMesh([2,1, -2,1, 0,-3],2);
	
	return true;
}
