/*File: SyntaxHighlighting.js
Customizable syntax and keyword highlighting for C-like languages.*/


var LANGUAGE_GLSL = {"void":"hl-t", "bool":"hl-t", "int":"hl-t", "uint":"hl-t", "float":"hl-t", "true":"hl-t", "false":"hl-t", "case":"hl-t", "default":"hl-t", "layout":"hl-t",
"usamplerCube":"hl-t",
"if":"hl-i", "else":"hl-i", "for":"hl-i", "while":"hl-i", "do":"hl-i", "return":"hl-i", "break":"hl-i", "continue":"hl-i", "switch":"hl-i", "const":"hl-i", "discard":"hl-i", "uniform":"hl-i",
"varying":"hl-i", "attribute":"hl-i", "in":"hl-i", "out":"hl-i", "inout":"hl-i",
"#if":"hl-f", "#ifdef":"hl-f", "#ifndef":"hl-f", "#include":"hl-f", "#else":"hl-f", "#endif":"hl-f", "#elif":"hl-f", "#define":"hl-f", "#undef":"hl-f"};

{
//add GLSL vector/matrix/sampler types of different sizes
var s = "hl-t", sampT = ["","u","i"],
	sampF = ["1D","2D","3D","Cube","2DRect","1DArray","2DArray","CubeArray","Buffer","2DMS","2DMSArray","1DShadow","2DShadow","CubeShadow","2DRectShadow","1DArrayShadow","2DArrayShadow","CubeArrayShadow"];
for (var i = 0; i < sampF.length; i++) {
	var str = "sampler"+sampF[i];
	for (var k = 0; k < 3; k++) LANGUAGE_GLSL[sampT[k]+str] = s;
}

for (var i = 2; i < 5; i++) {
	LANGUAGE_GLSL["vec"+i] = s;
	LANGUAGE_GLSL["ivec"+i] = s;
	LANGUAGE_GLSL["uvec"+i] = s;
	LANGUAGE_GLSL["bvec"+i] = s;
	LANGUAGE_GLSL["mat"+i] = s;
	for (var y = 2; y < 5; y++) LANGUAGE_GLSL["mat"+i+"x"+y] = s;
}

//add GLSL functions
var glslFuncs = ["acos","acosh","asin","asinh","atan","atanh","cos","cosh","degrees","radians","sin","sinh","tan","tanh","abs","ceil","clamp","dFdx","dFdy","exp","exp2","floor","fract","fwidth","inversesqrt","isinf","isnan","log","log2","max","min","mix","mod","modf","pow","round","roundEven","sign","smoothstep","sqrt","step","trunc","floatBitsToInt","intBitsToFloat","cross","distance","dot","equal","faceforward","length","normalize","notEqual","reflect","refract","all","any","greaterThan","greaterThanEqual","lessThan","lessThanEqual","not","texelFetch","texelFetchOffset","texture","textureGrad","textureGradOffset","textureLod","textureLodOffset","textureOffset","textureProj","textureProjGrad","textureProjGradOffset","textureProjLod","textureProjLodOffset","textureProjOffset","textureSize","determinant","inverse","matrixCompMult","outerProduct","transpose"];
for (var i = 0; i < glslFuncs.length; i++) LANGUAGE_GLSL[glslFuncs[i]] = "hl-f";

}


//character lookup for optimizing syntax parsing
var SH_CHAR_LUT, SH_HTML_LUT = new Array(7), SH_LUT_MAX = 127,
SH_ID_SINGLE_STRING = 1, SH_ID_DOUBLE_STRING = 2,
SH_ID_ARITHMETIC = 3, SH_ID_SCOPE = 4, SH_ID_SEPARATOR = 5,
SH_ID_SPACE = 6;



/*Function: InitializeSyntaxHighlighting
Initializes look up tables used by SyntaxHighlighting, must be called before using Highlight.*/
function InitializeSyntaxHighlighting() {
	SH_CHAR_LUT = new Uint8Array(SH_LUT_MAX);
	
	SH_CHAR_LUT["'".charCodeAt(0)] = SH_ID_SINGLE_STRING;
	SH_CHAR_LUT['"'.charCodeAt(0)] = SH_ID_DOUBLE_STRING;
	
	function enter(chars,id) {
		for (var i = 0; i < chars.length; i++) SH_CHAR_LUT[chars[i].charCodeAt(0)] = id;
	}
	enter(["=","+","-","*","/","&","^","|","!","%","~","%","<",">"], SH_ID_ARITHMETIC);
	enter(["[","]","{","}",";",":"], SH_ID_SCOPE);
	enter([",",".","(",")"], SH_ID_SEPARATOR);
	enter([" ","\t","\n","\r"], SH_ID_SPACE);
	
	
	SH_HTML_LUT[0] = null;
	SH_HTML_LUT[6] = null;
}




/*Class: SyntaxHighlighting
Syntax highlighting class, holds settings for styling of highlighting.

Constructor: SyntaxHighlighting
Construct new SyntaxHighlighting class with a blank configuration.
*/
/**@constructor*/
function SyntaxHighlighting(com,singStr,dblStr,arith,scope,sep,keywords) {
	/*Property: comment
	*String* HTML tag name for comment highlighting.*/
	this.comment = com;
	/*Property: singleQuoteString
	*String* HTML tag name for single quote string highlighting.*/
	this.singleQuoteString = singStr;
	/*Property: doubleQuoteString
	*String* HTML tag name for double quote string highlighting.*/
	this.doubleQuoteString = dblStr;
	/*Property: arithmetic
	*String* HTML tag name for arithmetic(= + - * & ^ | ! % ~ < >) highlighting.*/
	this.arithmetic = arith;
	/*Property: scope
	*String* HTML tag name for scope(; : { } [ ]) highlighting.*/
	this.scope = scope;
	/*Property: separator
	*String* HTML tag name for separator(, . ( )) highlighting.*/
	this.separator = sep;
	/*Property: keywords
	*object* Look up table for custom keyword highlighting, keys are strings(case sensitive) and values are HTML tag names.*/
	this.keywords = keywords?keywords:{};
}


/*Function: Highlight
Parse code of C-like language and generates HTML code with highlighting applied based off configuration.

Parameters:
*String* Code of C-like language.
*bool* escapeHtml - If true replaces < with &lt; to prevent HTML being parsed.

Returns:
*String* HTML code string.
*/
SyntaxHighlighting.prototype.Highlight = function(code, escapeHtml) {
	var tlen = code.length, html = "",
		buf = "", lastCh = null,
		inComment = 0, inString = 0,
		
		comment = this.comment?this.comment+">":null,
		singleString = this.singleQuoteString?true:false, doubleString = this.doubleQuoteString?true:false,
		arithmetic = this.arithmetic?true:false,
		keywords = this.keywords;
		
	SH_HTML_LUT[1] = this.singleQuoteString ? this.singleQuoteString+">" : null;
	SH_HTML_LUT[2] = this.doubleQuoteString ? this.doubleQuoteString+">" : null;
	SH_HTML_LUT[3] = this.arithmetic ? this.arithmetic+">" : null;
	SH_HTML_LUT[4] = this.scope ? this.scope+">" : null;
	SH_HTML_LUT[5] = this.separator ? this.separator+">" : null;
	
	for (var ti = 0; ti < tlen; ti++) {
		var c = code.charAt(ti);
	
		if (lastCh === "\\") {
			if (inString || inComment) html += c;
			else buf += c;
		
			if (c === "\\") c = "";
		
		} else {
	
			if (inComment) {
				if (inComment === 1) {
					if (c === "\n" || c === "\r") {
						if (comment) html += "</"+comment;
						inComment = 0;	
					}
					html += (escapeHtml&&c==="<"?"&lt;":c);
				
				} else {
					html += (escapeHtml&&c==="<"?"&lt;":c);
					if (lastCh === "*" && c === "/") {
						if (comment) html += "</"+comment;
						inComment = 0;
						continue;
					}
				}
			
			} else {
		
				if (inString) {
					html += (escapeHtml&&c==="<"?"&lt;":c);
					if (inString === 1) {
						if (c === "'") {
							if (singleString) html += "</"+SH_HTML_LUT[1];
							inString = 0;	
						}
				
					} else if (c === '"') {
						if (doubleString) html += "</"+SH_HTML_LUT[2];
						inString = 0;
					}
				
				} else {
	
					var chId = c.charCodeAt(0);
					
					if (lastCh === "/") {
						if (buf.length) {
							var keyStyle = keywords[buf];
							if (keyStyle) html += "<"+keyStyle+">"+buf+"</"+keyStyle+">";
							else html += buf;
							buf = "";
						}
						
						if (chId === 42) {//multi line comment /*, 42 = *
							inComment = 2;
							if (comment) html += "<"+comment+"/*";
							else html += "/*";
							lastCh = c;
							continue;
							
						} else if (chId === 47) {//single line comment, 47 = /
							inComment = 1;
							if (comment) html += "<"+comment+"//";
							else html += "//";
							lastCh = "";
							continue;
						
						} else {//no comment just insert '/' from last character
							if (arithmetic) html += "<"+SH_HTML_LUT[3]+"/"+"</"+SH_HTML_LUT[3];
							else html += "/";
						}
					}
				
					var type = (chId>=SH_LUT_MAX ? 0 : SH_CHAR_LUT[chId]);
			
					if (type) {
						if (buf.length) {
							var keyStyle = keywords[buf];
							if (keyStyle) html += "<"+keyStyle+">"+buf+"</"+keyStyle+">";
							else html += buf;
							buf = "";
						}
				
						var tag = SH_HTML_LUT[type];
						if (type < 3) {
							inString = type;
							if (tag) html += "<"+tag+c;
							else html += c;
					
						} else if (chId === 47) {
							//dont output / until next character so we can check if /* or //
							
						} else if (tag) html += "<"+tag+(escapeHtml&&chId===60?"&lt;":c)+"</"+tag;//60 = <
						else html += (escapeHtml&&chId===60?"&lt;":c);
				
					} else {
						buf += c;
					}
			
			
				}
			}
		}
	
		lastCh = c;
	}
	
	if (inComment && comment) html += "</"+comment;
	if (inString === 1 && singleString) html += "</"+SH_HTML_LUT[1];
	else if (inString === 2 && doubleString) html += "</"+SH_HTML_LUT[2];
	
	if (buf.length) {
		var keyStyle = keywords[buf];
		if (keyStyle) html += "<"+keyStyle+">"+buf+"</"+keyStyle+">";
		else html += buf;
	}
	
	return html;
}

