#version 150
precision highp float; precision highp int;
//Ethan Alexander Shulman 2023 "Artificial Neural Networks" February 20, 2023

in vec3 in_Position;
out vec4 uv;
out float time;

layout(std140) uniform ShaderviewBlock {
	vec4 mouse;
	float screenX,screenY;
	int frame,framerate;
};
layout(std140) uniform MidiBlock {
	vec4 midi[64];//x = note id, y = velocity, z = channel/track, w = time in seconds since note pressed
	int midiCount;
};
out vec4 music;//0.x = kick, 0.y = stick, 0.z = space bass, 0.w = super nova

/*LMMS midi converter https://www.lynxwave.com/LMMStoMIDI/LMMStoMIDIConverter.html since LMMS midi export is broken.

Launch options for midi config 'Shaderview console art/ArtificialNeuralNetworks.mid 7 0.4'
7 is the bitmask 0b000111 signifying which instruments are beats/drums with a fixed length.
0.4 is the fixed beat length in seconds.

Midi Instrument 0 = MaxV - KICKDRM1.wav
Midi Instrument 1 = MaxV - SIDESTIK.wav
Midi Instrument 2 = bass1.wav
Midi Instrument 3 = Fathom(Breath Click)
Midi Instrument 4 = Space Bass
Midi Instrument 5 = Supernova*/

#define TEMPO_TIME(bpm,note) (60.*4.*note/bpm)


#define ANIMATION 1
#define AA_SAMPLES 1


#define PI 3.141592653589793


mat2 r2d(float a) {
	float c = cos(a), s = sin(a);
	return mat2(c,s,-s,c);
}

//r2 hash http://extremelearning.com.au/unreasonable-effectiveness-of-quasirandom-sequences/
vec2 quasiRandom(float n) {
	#define R2V 1.32471795724474602596
	return fract(.5+n*(1./vec2(R2V,R2V*R2V)));
}


vec3 hue(float h) {
	return clamp(abs(mod(h*6.+vec3(0.,4.,2.),6.)-3.)-1.,0.,1.);
}



uniform sampler2D Buffer0;

void vert() {
	#ifdef ANIMATION
	time = float(frame)/float(framerate*AA_SAMPLES);
	music = vec4(0);
	for (int i = 0; i < midiCount; i++) {
		if (midi[i].z == 0.) {//kick
			music.x += max(0.,1.-midi[i].w/.4);
		} else if (midi[i].z == 1.) {//stick
			music.y += max(0.,1.-midi[i].w/.1);
		} else if (midi[i].z == 2.) {//bass drum
			music.x += max(0.,1.-midi[i].w/.3);
		} else if (midi[i].z == 4.) {//space bass
			music.z++;
		} else if (midi[i].z == 5.) {//super nova
			music.w += .1+.9*mod(midi[i].x,12.)/11.;
		}
	}
	music = min(music,1.);

	#else
	//screenshot 1
	time = 16.842;
	music = vec4(.6666,1,1,.9);
	
	//screenshot 2
	//time = 54.999;
	//music = vec4(1.,1,1,0.);
	#endif
	
	uv = vec4(in_Position.xy*vec2(screenX/screenY,1)+(quasiRandom(float(frame%AA_SAMPLES))-.5)*2./screenY,
		(in_Position.xy*.5+.5)*vec2(screenX,screenY));
	gl_Position = vec4(in_Position,1);
}
void frag() {
	float offX = clamp((time-20.)/30.,0.,1.)*.15, rot = clamp((time-5.)/45., 0.,1.);
	vec3 c = vec3(0);
	for (float l = 1.; l < 10.; l++) {
		vec2 p = uv.xy, offset = vec2(offX-l/10.*.1+music.w*l*.075, .04+music.x*.04);
		float trot = rot*time*.05*(.5+l*l*.03);
		for (int i = 0; i < 7; i++) {
			p = abs(p)-offset;
			p *= r2d(trot);
		}
		
		float wv = fract(time/TEMPO_TIME(140.,1./8.)-abs(p.x)*2.);
		wv = max(0., 1.-abs(wv-.5)*20.)*music.z;
		
		c += mix(vec3(.1),vec3(.2,.15,.05)*4.,wv)/(1.+max(0.,abs(p.y)-.001)*1000.)/l;//lines
		
		p.x = mod(abs(p.x),.1)-.05;
		float len = max(0.,length(p)-.008);
		c += vec3(.17,.13,.04)*(.5+music.y)/(1.+len*1000.)/l;
		c += vec3(.04,.4,.6)*wv*pow(max(0.,1.-len*50.),2.)/l;//neurons
	}
	
	if (time < 10.) c *= pow(time/10.,4.);
	
		
	vec4 sum = texelFetch(Buffer0,ivec2(uv.zw),0);
	if (frame%AA_SAMPLES == 0) sum = vec4(0);
	sum += vec4(c, 1.);
	gl_FragColor = sum;
}
